--# BoardModel
HEX_DIRS = {
    {0,-1},   -- north
    {-1,0},    -- northwest
    {-1,1},   -- southwest
    {0,1},    -- south
    {1,0},    -- southeast
    {1,-1},   -- northeast
}

EDGE_COLORS = {
    red = color(255,80,80),
    green = color(80,255,80),
    blue = color(80,80,255),
    orange= color(255,200,80),
    purple = color(180,80,255),
    teal = color(80,220,220)
}

BoardModel = class()

function BoardModel:init()
    self.hexCells = {}   -- key -> tile
end

function BoardModel:key(q,r)
    return q..","..r
end

function BoardModel:get(q,r)
    return self.hexCells[self:key(q,r)]
end

function BoardModel:set(q,r,tile)
    local key = self:key(q,r)
    self.hexCells[key] = {tile = tile,
        q = q,
        r = r
    }
end

function BoardModel:remove(q,r)
    self.hexCells[self:key(q,r)] = nil
end

function BoardModel:rotateTile(q, r, direction)
    local cell = self:get(q,r)
    if not cell then return end
    
    cell.tile:rotate(direction)
end

function BoardModel:allHexCellsAsFlatList()
    local list = {}
    for _,t in pairs(self.hexCells) do
        list[#list+1] = t
    end
    return list
end

function BoardModel:spawnInitialTiles()
    self:set(0,0, TileModel("A"))
    self:set(1,0, TileModel("B"))
    self:set(0,1, TileModel("C"))
    self:set(-1,0,TileModel("D"))
end

function BoardModel:playableHexes()
    local out = {}
    local seen = {}
    
    for _,cell in pairs(self.hexCells) do
        
        -- 👇 SKIP candidate tile
        if cell.tile.kind ~= "candidate" then
            
            for _,d in ipairs(HEX_DIRS) do
                local q = cell.q + d[1]
                local r = cell.r + d[2]
                local k = self:key(q,r)
                
                if not self.hexCells[k] and not seen[k] then
                    seen[k] = true
                    out[#out+1] = {q=q, r=r}
                end
            end
            
        end
    end
    
    return out
end

function BoardModel:moveCandidateTo(q,r)
  local candTile = nil
  
  -- remove old candidate but KEEP the tile object
  for k,cell in pairs(self.hexCells) do
    if cell.tile and cell.tile.kind == "candidate" then
      candTile = cell.tile
      self.hexCells[k] = nil
      break
    end
  end
  
  -- if none exists yet, create it once
  if not candTile then
    candTile = TileModel("CAND", nil, "candidate")
  end
  
  -- place SAME tile at new spot (preserves sides + rotation state)
  self:set(q,r,candTile)
  
  self:testCandidateAgainstNeighbors()
end

function BoardModel:countCandidateMatches()
    local cand
    
    for _,cell in ipairs(self:allHexCellsAsFlatList()) do
        if cell.tile and cell.tile.kind == "candidate" then
            cand = cell
            break
        end
    end
    
    if not cand then return 0 end
    
    local count = 0
    
    for i,d in ipairs(HEX_DIRS) do
        local nq = cand.q + d[1]
        local nr = cand.r + d[2]
        
        local neighbor = self:get(nq,nr)
        if neighbor and neighbor.tile then
            local mySide = DIR_TO_SIDE[i]
            
            local oppDir = ((i+2)%6)+1
            local theirSide = DIR_TO_SIDE[oppDir]
            
            if cand.tile.sides[mySide] ==
            neighbor.tile.sides[theirSide] then
                count = count + 1
            end
        end
    end
    
    return count
end

function BoardModel:debugCandidateContacts()
    local cand
    
    for _,cell in ipairs(self:allHexCellsAsFlatList()) do
        if cell.tile and cell.tile.kind == "candidate" then
            cand = cell
            break
        end
    end
    
    if not cand then
        print("NO CANDIDATE")
        return
    end
    
    print("=== DEBUG CONTACTS at", cand.q, cand.r, "===")
    
    for i,d in ipairs(HEX_DIRS) do
        local nq = cand.q + d[1]
        local nr = cand.r + d[2]
        
        local neighbor = self:get(nq,nr)
        if neighbor and neighbor.tile then
            local myColor = cand.tile.sides[i]
            
            -- your *actual* facing side math
            local opp = ((i+2)%6)+1
            local theirColor = neighbor.tile.sides[opp]
            
            local ok = (myColor == theirColor)
            
            print(
            "DIR", i,
            "cand side", i, myColor,
            "| neigh at", nq, nr,
            "side", opp, theirColor,
            ok and "MATCH" or "NO"
            )
        end
    end
end

function BoardModel:testCandidateAgainstNeighbors()
    local cand
    
    for _,cell in ipairs(self:allHexCellsAsFlatList()) do
        if cell.tile.kind == "candidate" then
            cand = cell
            break
        end
    end
    
    if not cand then
        print("NO CANDIDATE")
        return
    end
    
    for i,d in ipairs(HEX_DIRS) do
        local nq = cand.q + d[1]
        local nr = cand.r + d[2]
        
        local neigh = self:get(nq,nr)
        if neigh then
            print("---- comparing to neighbor at", nq, nr)
            runTileMatchTest(cand.tile, neigh.tile)
        end
    end
end

function runTileMatchTest(a, b)
    print("=== TILE MATCH TEST ===")
    print("A:", table.unpack(a.sides))
    print("B:", table.unpack(b.sides))
    
    local pass = true
    
    for i=1,6 do
        local opp = ((i+2)%6)+1
        local ok = a.sides[i] == b.sides[opp]
        
        print(
        "side", i,
        "vs", opp,
        ok and "PASS" or "FAIL"
        )
        
        if not ok then pass = false end
    end
    
    print(pass and "ALL PASS" or "SOME FAIL")
end





